<?php
require_once "../includes/config.php";

if (!isset($_SESSION['user_level']) || $_SESSION['user_level'] !== 'admin') {
    header("Location: login.php");
    exit;
}

$error = "";
$success = "";
$debug_info = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Enable error reporting for debugging
    mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
    
    try {
        $nama_kayu = isset($_POST['nama_kayu']) ? trim($_POST['nama_kayu']) : '';
        $kekuatan_kayu = isset($_POST['kekuatan_kayu']) ? floatval($_POST['kekuatan_kayu']) : 0;
        $kepadatan_kayu = isset($_POST['kepadatan_kayu']) ? floatval($_POST['kepadatan_kayu']) : 0;
        $kadar_air = isset($_POST['kadar_air']) ? floatval($_POST['kadar_air']) : 0;
        $harga = isset($_POST['harga']) ? floatval($_POST['harga']) : 0;

        // Validasi input
        $errors = [];
        
        if (empty($nama_kayu)) {
            $errors[] = "Nama kayu tidak boleh kosong.";
        }
        
        if (strlen($nama_kayu) > 100) {
            $errors[] = "Nama kayu terlalu panjang (maksimal 100 karakter).";
        }
        
        if ($kekuatan_kayu <= 0) {
            $errors[] = "Kekuatan kayu harus diisi dengan angka positif.";
        }
        
        if ($kepadatan_kayu <= 0) {
            $errors[] = "Kepadatan kayu harus diisi dengan angka positif.";
        }
        
        if ($kadar_air <= 0) {
            $errors[] = "Kadar air harus diisi dengan angka positif.";
        }
        
        if ($harga <= 0) {
            $errors[] = "Harga harus diisi dengan angka positif.";
        }

        if (!empty($errors)) {
            $error = implode("<br>", $errors);
        } else {
            // Cek duplikasi nama kayu
            $check_stmt = $conn->prepare("SELECT COUNT(*) FROM kayu WHERE nama_kayu = ? AND id_kayu != ?");
            $check_id = isset($_POST['id_kayu']) ? $_POST['id_kayu'] : 0;
            $check_stmt->bind_param("si", $nama_kayu, $check_id);
            $check_stmt->execute();
            $check_stmt->bind_result($count);
            $check_stmt->fetch();
            $check_stmt->close();
            
            if ($count > 0) {
                $error = "Nama kayu sudah ada dalam database.";
            } else {
                if (isset($_POST['id_kayu']) && is_numeric($_POST['id_kayu']) && $_POST['id_kayu'] > 0) {
                    // Update existing kayu
                    $stmt = $conn->prepare("UPDATE kayu SET nama_kayu = ?, kekuatan_kayu = ?, kepadatan_kayu = ?, kadar_air = ?, harga = ? WHERE id_kayu = ?");
                    $stmt->bind_param("sdddii", $nama_kayu, $kekuatan_kayu, $kepadatan_kayu, $kadar_air, $harga, $_POST['id_kayu']);
                    
                    if ($stmt->execute()) {
                        if ($stmt->affected_rows > 0) {
                            $success = "Data kayu berhasil diperbarui.";
                        } else {
                            $success = "Tidak ada perubahan pada data kayu.";
                        }
                    } else {
                        $error = "Gagal memperbarui data kayu: " . $stmt->error;
                    }
                    $stmt->close();
                } else {
                    // Insert new kayu
                    $stmt = $conn->prepare("INSERT INTO kayu (nama_kayu, kekuatan_kayu, kepadatan_kayu, kadar_air, harga) VALUES (?, ?, ?, ?, ?)");
                    $stmt->bind_param("sdddd", $nama_kayu, $kekuatan_kayu, $kepadatan_kayu, $kadar_air, $harga);
                    
                    if ($stmt->execute()) {
                        $success = "Data kayu berhasil ditambahkan dengan ID: " . $stmt->insert_id;
                    } else {
                        $error = "Gagal menambahkan data kayu: " . $stmt->error;
                    }
                    $stmt->close();
                }
            }
        }
        
    } catch (mysqli_sql_exception $e) {
        $error = "Database Error: " . $e->getMessage();
        $debug_info[] = "Error Code: " . $e->getCode();
        $debug_info[] = "Error Message: " . $e->getMessage();
    } catch (Exception $e) {
        $error = "Error: " . $e->getMessage();
        $debug_info[] = "Error: " . $e->getMessage();
    }
}

// For edit: load data if edit parameter is set
$edit_kayu = null;
if (isset($_GET['edit']) && is_numeric($_GET['edit'])) {
    $stmt = $conn->prepare("SELECT id_kayu, nama_kayu, kekuatan_kayu, kepadatan_kayu, kadar_air, harga FROM kayu WHERE id_kayu = ?");
    $stmt->bind_param("i", $_GET['edit']);
    $stmt->execute();
    $stmt->bind_result($id_kayu, $nama_kayu, $kekuatan_kayu, $kepadatan_kayu, $kadar_air, $harga);
    if ($stmt->fetch()) {
        $edit_kayu = [
            'id_kayu' => $id_kayu,
            'nama_kayu' => $nama_kayu,
            'kekuatan_kayu' => $kekuatan_kayu,
            'kepadatan_kayu' => $kepadatan_kayu,
            'kadar_air' => $kadar_air,
            'harga' => $harga
        ];
    }
    $stmt->close();
}

// For delete
if (isset($_GET['delete']) && is_numeric($_GET['delete'])) {
    $del_id = $_GET['delete'];
    $stmt = $conn->prepare("DELETE FROM kayu WHERE id_kayu = ?");
    $stmt->bind_param("i", $del_id);
    $stmt->execute();
    $stmt->close();
    header("Location: input_data.php");
    exit;
}

// Fetch all kayu
$kayu_list = [];
$result = $conn->query("SELECT * FROM kayu ORDER BY id_kayu DESC");
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $kayu_list[] = $row;
    }
    $result->free();
}

include "../includes/header.php";
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Kelola Data Kayu</title>
    <style>
        .error-msg {
            background-color: #f8d7da;
            color: #721c24;
            padding: 10px;
            border-radius: 4px;
            margin-bottom: 15px;
            border: 1px solid #f5c6cb;
        }
        .success-msg {
            background-color: #d4edda;
            color: #155724;
            padding: 10px;
            border-radius: 4px;
            margin-bottom: 15px;
            border: 1px solid #c3e6cb;
        }
        .debug-info {
            background-color: #fff3cd;
            color: #856404;
            padding: 10px;
            border-radius: 4px;
            margin-bottom: 15px;
            border: 1px solid #ffeaa7;
            font-family: monospace;
            font-size: 12px;
        }
        .form-group {
            margin-bottom: 15px;
        }
        label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
        }
        input[type="text"],
        input[type="number"] {
            width: 100%;
            padding: 8px;
            border: 1px solid #ddd;
            border-radius: 4px;
            box-sizing: border-box;
        }
        button {
            background-color: #007bff;
            color: white;
            padding: 10px 20px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
        }
        button:hover {
            background-color: #0056b3;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }
        th, td {
            padding: 10px;
            text-align: left;
            border-bottom: 1px solid #ddd;
        }
        th {
            background-color: #f8f9fa;
            font-weight: bold;
        }
        .actions a {
            margin-right: 10px;
            text-decoration: none;
        }
        .actions a.edit {
            color: #007bff;
        }
        .actions a.delete {
            color: #dc3545;
        }
    </style>
</head>
<body>
    <h2>Kelola Data Kayu</h2>

    <?php if (!empty($debug_info)): ?>
        <div class="debug-info">
            <strong>Debug Info:</strong><br>
            <?php echo implode("<br>", $debug_info); ?>
        </div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="error-msg"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>
    <?php if ($success): ?>
        <div class="success-msg"><?php echo htmlspecialchars($success); ?></div>
    <?php endif; ?>

    <form method="POST" action="" class="compact-form">
        <input type="hidden" name="id_kayu" value="<?php echo $edit_kayu ? $edit_kayu['id_kayu'] : ''; ?>">
        
        <div class="form-group">
            <label for="nama_kayu">Nama Kayu:</label>
            <input type="text" name="nama_kayu" id="nama_kayu" required 
                   value="<?php echo $edit_kayu ? htmlspecialchars($edit_kayu['nama_kayu']) : ''; ?>"
                   maxlength="100">
        </div>
        
        <div class="form-group">
            <label for="kekuatan_kayu">Kekuatan (MPa):</label>
            <input type="number" step="0.01" name="kekuatan_kayu" id="kekuatan_kayu" required 
                   value="<?php echo $edit_kayu ? htmlspecialchars($edit_kayu['kekuatan_kayu']) : ''; ?>"
                   min="0" max="1000">
        </div>
        
        <div class="form-group">
            <label for="kepadatan_kayu">Kepadatan (kg/m³):</label>
            <input type="number" step="0.01" name="kepadatan_kayu" id="kepadatan_kayu" required 
                   value="<?php echo $edit_kayu ? htmlspecialchars($edit_kayu['kepadatan_kayu']) : ''; ?>"
                   min="0" max="2000">
        </div>
        
        <div class="form-group">
            <label for="kadar_air">Kadar Air (%):</label>
            <input type="number" step="0.01" name="kadar_air" id="kadar_air" required 
                   value="<?php echo $edit_kayu ? htmlspecialchars($edit_kayu['kadar_air']) : ''; ?>"
                   min="0" max="100">
        </div>
        
        <div class="form-group">
            <label for="harga">Harga (IDR/m³):</label>
            <input type="number" step="0.01" name="harga" id="harga" required 
                   value="<?php echo $edit_kayu ? htmlspecialchars($edit_kayu['harga']) : ''; ?>"
                   min="0" max="999999999">
        </div>
        
        <div class="form-actions">
            <button type="submit"><?php echo $edit_kayu ? 'Update' : 'Tambah'; ?></button>
            <?php if ($edit_kayu): ?>
                <a href="input_data.php" style="margin-left: 10px;">Batal</a>
            <?php endif; ?>
        </div>
    </form>

    <hr>

    <h3>Daftar Kayu</h3>
    <table border="1" cellpadding="8" cellspacing="0">
        <thead>
            <tr>
                <th>ID</th>
                <th>Nama Kayu</th>
                <th>Kekuatan (MPa)</th>
                <th>Kepadatan (kg/m³)</th>
                <th>Kadar Air (%)</th>
                <th>Harga (IDR/m³)</th>
                <th>Aksi</th>
            </tr>
        </thead>
        <tbody>
            <?php if (count($kayu_list) > 0): ?>
                <?php foreach ($kayu_list as $kayu): ?>
                    <tr>
                        <td><?php echo isset($kayu['id_kayu']) ? $kayu['id_kayu'] : ''; ?></td>
                        <td><?php echo isset($kayu['nama_kayu']) ? htmlspecialchars($kayu['nama_kayu']) : ''; ?></td>
                        <td><?php echo isset($kayu['kekuatan_kayu']) && $kayu['kekuatan_kayu'] !== null ? number_format(floatval($kayu['kekuatan_kayu']), 2, ',', '.') : '0'; ?></td>
                        <td><?php echo isset($kayu['kepadatan_kayu']) && $kayu['kepadatan_kayu'] !== null ? number_format(floatval($kayu['kepadatan_kayu']), 2, ',', '.') : '0'; ?></td>
                        <td><?php echo isset($kayu['kadar_air']) && $kayu['kadar_air'] !== null ? number_format(floatval($kayu['kadar_air']), 2, ',', '.') : '0'; ?></td>
                        <td><?php echo isset($kayu['harga']) && $kayu['harga'] !== null ? number_format(floatval($kayu['harga']), 0, ',', '.') : '0'; ?></td>
                        <td class="actions">
                            <a href="input_data.php?edit=<?php echo isset($kayu['id_kayu']) ? $kayu['id_kayu'] : ''; ?>" class="edit">Edit</a> |
                            <a href="input_data.php?delete=<?php echo isset($kayu['id_kayu']) ? $kayu['id_kayu'] : ''; ?>" 
                               onclick="return confirm('Yakin ingin menghapus kayu ini?');" class="delete">Hapus</a>
                        </td>
                    </tr>
                <?php endforeach;?>
            <?php else: ?>
                <tr><td colspan="7" style="text-align: center;">Belum ada data kayu.</td></tr>
            <?php endif; ?>
        </tbody>
    </table>

    <p style="margin-top: 20px;">
        <a href="../diagnostic.php" target="_blank">Cek Diagnostic Database</a>
    </p>
</body>
</html>

<?php
include "../includes/footer.php";
?>
