<?php
/* Halaman untuk menampilkan hasil rekomendasi dengan perhitungan ELECTRE menggunakan matriks concordance dan discordance */
require_once "../includes/config.php";

if (!isset($_SESSION['user_level'])) {
    header("Location: login.php");
    exit;
}

// Function to fetch criteria and their details
function getCriteria($conn) {
    $criteria = [];
    $sql = "SELECT k.id_kriteria, k.nama_kriteria, k.tipe, b.nilai_bobot
            FROM kriteria k
            JOIN bobot b ON k.id_kriteria = b.id_kriteria
            ORDER BY k.id_kriteria";
    $result = $conn->query($sql);
    while ($row = $result->fetch_assoc()) {
        $criteria[$row['id_kriteria']] = $row;
    }
    return $criteria;
}

// Function to fetch wood alternatives with criteria values
function getKayu($conn) {
    $kayus = [];
    $sql = "SELECT id_kayu, nama_kayu, kekuatan_kayu, kepadatan_kayu, kadar_air, harga FROM kayu ORDER BY id_kayu";
    $result = $conn->query($sql);
    while ($row = $result->fetch_assoc()) {
        $kayus[$row['id_kayu']] = $row;
    }
    return $kayus;
}

// Normalize decision matrix
function normalizeDecisionMatrix($penilaian, $criteria) {
    $normalized = [];
    foreach ($criteria as $cid => $crit) {
        $values = [];
        foreach ($penilaian as $id_kayu => $vals) {
            $val = isset($vals[$cid]) ? $vals[$cid] : 0;
            if ($val > 0) {
                $values[] = $val;
            }
        }
        $maxVal = max($values);
        $minVal = min($values);
        if ($maxVal == 0) $maxVal = 1;
        if ($minVal == 0) $minVal = 1;
        
        foreach ($penilaian as $id_kayu => $vals) {
            $val = isset($vals[$cid]) ? $vals[$cid] : 0;
            if ($val == 0) $val = 1;
            
            if ($crit['tipe'] == 'Benefit') {
                $normalized[$id_kayu][$cid] = $val / $maxVal;
            } else {
                $normalized[$id_kayu][$cid] = $minVal / $val;
            }
        }
    }
    return $normalized;
}

// Calculate weighted normalized matrix
function weightedNormalizedMatrix($normalized, $criteria) {
    $weighted = [];
    foreach ($normalized as $id_kayu => $values) {
        foreach ($values as $cid => $value) {
            $weighted[$id_kayu][$cid] = $value * $criteria[$cid]['nilai_bobot'];
        }
    }
    return $weighted;
}

// Calculate concordance and discordance indices
function calculateConcordanceDiscordance($weighted, $criteria, $kayus) {
    $concordance = [];
    $discordance = [];
    $ids = array_keys($kayus);
    
    foreach ($ids as $i) {
        foreach ($ids as $j) {
            if ($i == $j) {
                $concordance[$i][$j] = [];
                $discordance[$i][$j] = [];
                continue;
            }
            
            $concordance[$i][$j] = [];
            $discordance[$i][$j] = [];
            
            foreach ($criteria as $cid => $crit) {
                if ($weighted[$i][$cid] >= $weighted[$j][$cid]) {
                    $concordance[$i][$j][] = $cid;
                } else {
                    $discordance[$i][$j][] = $cid;
                }
            }
        }
    }
    
    return ['concordance' => $concordance, 'discordance' => $discordance];
}

// Calculate concordance matrix
function concordanceMatrix($weighted, $criteria, $kayus) {
    $C = [];
    $ids = array_keys($kayus);
    
    foreach ($ids as $i) {
        foreach ($ids as $j) {
            if ($i == $j) {
                $C[$i][$j] = 0;
                continue;
            }
            
            $sum_bobot = 0;
            foreach ($criteria as $cid => $crit) {
                if ($weighted[$i][$cid] >= $weighted[$j][$cid]) {
                    $sum_bobot += $crit['nilai_bobot'];
                }
            }
            $C[$i][$j] = $sum_bobot;
        }
    }
    return $C;
}

// Calculate discordance matrix - CORRECTED: Using proper ELECTRE-I formula
function discordanceMatrix($weighted, $criteria, $kayus) {
    $D = [];
    $ids = array_keys($kayus);
    
    foreach ($ids as $i) {
        foreach ($ids as $j) {
            if ($i == $j) {
                $D[$i][$j] = 0;
                continue;
            }
            
            // Find maximum difference across all criteria for denominator
            $max_denominator = 0;
            foreach ($criteria as $cid => $crit) {
                $diff = abs($weighted[$i][$cid] - $weighted[$j][$cid]);
                $max_denominator = max($max_denominator, $diff);
            }
            
            // Calculate discordance using ELECTRE-I formula
            $discordance_values = [];
            foreach ($criteria as $cid => $crit) {
                if ($weighted[$i][$cid] < $weighted[$j][$cid]) {
                    $diff = abs($weighted[$i][$cid] - $weighted[$j][$cid]);
                    if ($max_denominator > 0) {
                        $discordance_values[] = $diff / $max_denominator;
                    }
                }
            }
            
            $D[$i][$j] = !empty($discordance_values) ? max($discordance_values) : 0;
        }
    }
    return $D;
}

// Calculate dominant matrices
function calculateDominantMatrices($C, $D, $kayus) {
    // Calculate thresholds
    $maxC = 0;
    $minC = PHP_FLOAT_MAX;
    $maxD = 0;
    $minD = PHP_FLOAT_MAX;
    
    $ids = array_keys($kayus);
    foreach ($ids as $i) {
        foreach ($ids as $j) {
            if ($i != $j) {
                $maxC = max($maxC, $C[$i][$j]);
                $minC = min($minC, $C[$i][$j]);
                $maxD = max($maxD, $D[$i][$j]);
                $minD = min($minD, $D[$i][$j]);
            }
        }
    }
    
    $thresholdC = ($maxC + $minC) / 2;
    $thresholdD = ($maxD + $minD) / 2;
    
    // Calculate dominant matrices
    $F = []; // Concordance dominant
    $G = []; // Discordance dominant
    
    foreach ($ids as $i) {
        foreach ($ids as $j) {
            if ($i == $j) {
                $F[$i][$j] = '-';
                $G[$i][$j] = '-';
            } else {
                $F[$i][$j] = ($C[$i][$j] >= $thresholdC) ? 1 : 0;
                $G[$i][$j] = ($D[$i][$j] <= $thresholdD) ? 1 : 0;
            }
        }
    }
    
    // Calculate aggregate matrix
    $E = [];
    foreach ($ids as $i) {
        foreach ($ids as $j) {
            if ($i == $j) {
                $E[$i][$j] = '-';
            } else {
                $E[$i][$j] = ($F[$i][$j] == 1 && $G[$i][$j] == 1) ? 1 : 0;
            }
        }
    }
    
    return [
        'F' => $F,
        'G' => $G,
        'E' => $E,
        'thresholdC' => $thresholdC,
        'thresholdD' => $thresholdD
    ];
}

// Calculate final scores
function calculateFinalScores($E, $kayus) {
    $scores = [];
    $ids = array_keys($kayus);
    
    foreach ($ids as $i) {
        $count = 0;
        foreach ($ids as $j) {
            if ($i != $j && isset($E[$i][$j]) && $E[$i][$j] == 1) {
                $count++;
            }
        }
        $scores[$i] = $count;
    }
    
    arsort($scores);
    return $scores;
}

/* Step 1: Get data */
$criteria = getCriteria($conn);
$kayus = getKayu($conn);

// Prepare penilaian array from kayu data columns
$penilaian = [];
foreach ($kayus as $id_kayu => $kayu) {
    $penilaian[$id_kayu] = [
        1 => $kayu['kekuatan_kayu'],
        2 => $kayu['kepadatan_kayu'],
        3 => $kayu['kadar_air'],
        4 => $kayu['harga']
    ];
}

/* Step 2: Normalize */
$normalized = normalizeDecisionMatrix($penilaian, $criteria);

/* Step 3: Calculate weighted normalized matrix */
$weighted = weightedNormalizedMatrix($normalized, $criteria);

/* Step 4: Calculate concordance and discordance matrices */
$cd_indices = calculateConcordanceDiscordance($weighted, $criteria, $kayus);
$C = concordanceMatrix($weighted, $criteria, $kayus);
$D = discordanceMatrix($weighted, $criteria, $kayus);

/* Step 5: Calculate dominant matrices */
$dominant = calculateDominantMatrices($C, $D, $kayus);

/* Step 6: Calculate final scores */
$scores = calculateFinalScores($dominant['E'], $kayus);

// Step 7: Store scores and rank into hasil_electre table
$conn->query("TRUNCATE TABLE hasil_electre");
$rank = 1;
foreach ($scores as $id_kayu => $score) {
    $stmt = $conn->prepare("INSERT INTO hasil_electre (id_kayu, skor, peringkat) VALUES (?, ?, ?)");
    $stmt->bind_param("idi", $id_kayu, $score, $rank);
    $stmt->execute();
    $stmt->close();
    $rank++;
}

// Fetch and display results ordered by peringkat ascending
$stmt = $conn->prepare("
    SELECT h.id_hasil, k.id_kayu, k.nama_kayu, h.skor, h.peringkat 
    FROM hasil_electre h 
    JOIN kayu k ON h.id_kayu = k.id_kayu 
    ORDER BY h.peringkat ASC
");
$stmt->execute();
$stmt->bind_result($id_hasil, $id_kayu, $nama_kayu, $skor, $peringkat);

// Store results for later use
$results = [];
while ($stmt->fetch()) {
    $results[] = [
        'id_hasil' => $id_hasil,
        'id_kayu' => $id_kayu,
        'nama_kayu' => $nama_kayu,
        'skor' => $skor,
        'peringkat' => $peringkat
    ];
}

include "../includes/header.php";
?>

<h2>Hasil Perhitungan Metode Electre</h2>

<!-- Data Nilai Awal -->
<h3>Data Nilai Awal</h3>
<table class="dashboard-table">
    <thead>
        <tr>
            <th>No.</th>
            <th>Kode</th>
            <th>Alternatif</th>
            <th colspan="4">Nilai Kriteria</th>
        </tr>
        <tr>
            <th colspan="3"></th>
            <th>Kekuatan</th>
            <th>Kepadatan</th>
            <th>Kadar Air</th>
            <th>Harga</th>
        </tr>
    </thead>
    <tbody>
        <?php 
        $no = 1;
        foreach ($kayus as $id_kayu => $kayu): 
        ?>
            <tr>
                <td><?php echo $no++; ?></td>
                <td>K00<?php echo $id_kayu; ?></td>
                <td><?php echo htmlspecialchars($kayu['nama_kayu']); ?></td>
                <td><?php echo $kayu['kekuatan_kayu']; ?></td>
                <td><?php echo $kayu['kepadatan_kayu']; ?></td>
                <td><?php echo $kayu['kadar_air']; ?></td>
                <td><?php echo $kayu['harga']; ?></td>
            </tr>
        <?php endforeach; ?>
    </tbody>
</table>

<!-- Matriks Normalisasi -->
<h3>Matriks Normalisasi</h3>
<table class="dashboard-table">
    <thead>
        <tr>
            <th>No.</th>
            <th>Kode</th>
            <th>Alternatif</th>
            <th colspan="4">Nilai Normalisasi</th>
        </tr>
        <tr>
            <th colspan="3"></th>
            <th>Kekuatan</th>
            <th>Kepadatan</th>
            <th>Kadar Air</th>
            <th>Harga</th>
        </tr>
    </thead>
    <tbody>
        <?php 
        $no = 1;
        foreach ($kayus as $id_kayu => $kayu): 
        ?>
            <tr>
                <td><?php echo $no++; ?></td>
                <td>K00<?php echo $id_kayu; ?></td>
                <td><?php echo htmlspecialchars($kayu['nama_kayu']); ?></td>
                <td><?php echo number_format($normalized[$id_kayu][1], 4, ',', '.'); ?></td>
                <td><?php echo number_format($normalized[$id_kayu][2], 4, ',', '.'); ?></td>
                <td><?php echo number_format($normalized[$id_kayu][3], 4, ',', '.'); ?></td>
                <td><?php echo number_format($normalized[$id_kayu][4], 4, ',', '.'); ?></td>
            </tr>
        <?php endforeach; ?>
    </tbody>
</table>

<!-- Matriks Normalisasi Terbobot -->
<h3>Matriks Normalisasi Terbobot</h3>
<table class="dashboard-table">
    <thead>
        <tr>
            <th>No.</th>
            <th>Kode</th>
            <th>Alternatif</th>
            <th colspan="4">Nilai Normalisasi Terbobot</th>
        </tr>
        <tr>
            <th colspan="3"></th>
            <th>Kekuatan</th>
            <th>Kepadatan</th>
            <th>Kadar Air</th>
            <th>Harga</th>
        </tr>
    </thead>
    <tbody>
        <?php 
        $no = 1;
        foreach ($kayus as $id_kayu => $kayu): 
        ?>
            <tr>
                <td><?php echo $no++; ?></td>
                <td>K00<?php echo $id_kayu; ?></td>
                <td><?php echo htmlspecialchars($kayu['nama_kayu']); ?></td>
                <td><?php echo number_format($weighted[$id_kayu][1], 4, ',', '.'); ?></td>
                <td><?php echo number_format($weighted[$id_kayu][2], 4, ',', '.'); ?></td>
                <td><?php echo number_format($weighted[$id_kayu][3], 4, ',', '.'); ?></td>
                <td><?php echo number_format($weighted[$id_kayu][4], 4, ',', '.'); ?></td>
            </tr>
        <?php endforeach; ?>
    </tbody>
</table>

<!-- Himpunan Concordance dan Discordance index -->
<h3>Himpunan Concordance dan Discordance index</h3>
<table class="dashboard-table">
    <thead>
        <tr>
            <th>No.</th>
            <th>Alternatif 1</th>
            <th>Alternatif 2</th>
            <th>Index Concordance</th>
            <th>Index Discordance</th>
        </tr>
    </thead>
    <tbody>
        <?php 
        $no = 1;
        $ids = array_keys($kayus);
        foreach ($ids as $i) {
            foreach ($ids as $j) {
                if ($i != $j) {
                    $concordance_str = implode(', ', $cd_indices['concordance'][$i][$j]);
                    $discordance_str = implode(', ', $cd_indices['discordance'][$i][$j]);
                    if (empty($concordance_str)) $concordance_str = '-';
                    if (empty($discordance_str)) $discordance_str = '-';
        ?>
            <tr>
                <td><?php echo $no++; ?></td>
                <td>K00<?php echo $i; ?>-<?php echo htmlspecialchars($kayus[$i]['nama_kayu']); ?></td>
                <td>K00<?php echo $j; ?>-<?php echo htmlspecialchars($kayus[$j]['nama_kayu']); ?></td>
                <td><?php echo $concordance_str; ?></td>
                <td><?php echo $discordance_str; ?></td>
            </tr>
        <?php 
                }
            }
        }
        ?>
    </tbody>
</table>

<!-- Matriks Concordance -->
<h3>Matriks Concordance</h3>
<table class="dashboard-table">
    <thead>
        <tr>
            <th>C</th>
            <?php foreach ($kayus as $id_kayu => $kayu): ?>
                <th>K00<?php echo $id_kayu; ?></th>
            <?php endforeach; ?>
        </tr>
    </thead>
    <tbody>
        <?php 
        foreach ($kayus as $i => $kayu_i): 
        ?>
            <tr>
                <th>K00<?php echo $i; ?></th>
                <?php 
                foreach ($kayus as $j => $kayu_j): 
                    $value = ($i == $j) ? '-' : number_format($C[$i][$j], 2, ',', '.');
                ?>
                    <td><?php echo $value; ?></td>
                <?php endforeach; ?>
            </tr>
        <?php endforeach; ?>
    </tbody>
</table>

<!-- Matriks Discordance -->
<h3>Matriks Discordance</h3>
<table class="dashboard-table">
    <thead>
        <tr>
            <th>D</th>
            <?php foreach ($kayus as $id_kayu => $kayu): ?>
                <th>K00<?php echo $id_kayu; ?></th>
            <?php endforeach; ?>
        </tr>
    </thead>
    <tbody>
        <?php 
        foreach ($kayus as $i => $kayu_i): 
        ?>
            <tr>
                <th>K00<?php echo $i; ?></th>
                <?php 
                foreach ($kayus as $j => $kayu_j): 
                    $value = ($i == $j) ? '-' : number_format($D[$i][$j], 4, ',', '.');
                ?>
                    <td><?php echo $value; ?></td>
                <?php endforeach; ?>
            </tr>
        <?php endforeach; ?>
    </tbody>
</table>

<!-- Matriks Dominan Concordance -->
<h3>Matriks Dominan Concordance | Nilai ambang (Threshold) : <?php echo number_format($dominant['thresholdC'], 4, ',', '.'); ?></h3>
<table class="dashboard-table">
    <thead>
        <tr>
            <th>F</th>
            <?php foreach ($kayus as $id_kayu => $kayu): ?>
                <th>K00<?php echo $id_kayu; ?></th>
            <?php endforeach; ?>
        </tr>
    </thead>
    <tbody>
        <?php 
        foreach ($kayus as $i => $kayu_i): 
        ?>
            <tr>
                <th>K00<?php echo $i; ?></th>
                <?php 
                foreach ($kayus as $j => $kayu_j): 
                    $value = $dominant['F'][$i][$j];
                ?>
                    <td><?php echo $value; ?></td>
                <?php endforeach; ?>
            </tr>
        <?php endforeach; ?>
    </tbody>
</table>

<!-- Matriks Dominan Discordance -->
<h3>Matriks Dominan Discordance | Nilai ambang (Threshold) : <?php echo number_format($dominant['thresholdD'], 4, ',', '.'); ?></h3>
<table class="dashboard-table">
    <thead>
        <tr>
            <th>G</th>
            <?php foreach ($kayus as $id_kayu => $kayu): ?>
                <th>K00<?php echo $id_kayu; ?></th>
            <?php endforeach; ?>
        </tr>
    </thead>
    <tbody>
        <?php 
        foreach ($kayus as $i => $kayu_i): 
        ?>
            <tr>
                <th>K00<?php echo $i; ?></th>
                <?php 
                foreach ($kayus as $j => $kayu_j): 
                    $value = $dominant['G'][$i][$j];
                ?>
                    <td><?php echo $value; ?></td>
                <?php endforeach; ?>
            </tr>
        <?php endforeach; ?>
    </tbody>
</table>

<!-- Matriks Aggregate Dominance -->
<h3>Matriks Aggregate Dominance</h3>
<table class="dashboard-table">
    <thead>
        <tr>
            <th>E</th>
            <?php foreach ($kayus as $id_kayu => $kayu): ?>
                <th>K00<?php echo $id_kayu; ?></th>
            <?php endforeach; ?>
        </tr>
    </thead>
    <tbody>
        <?php 
        foreach ($kayus as $i => $kayu_i): 
        ?>
            <tr>
                <th>K00<?php echo $i; ?></th>
                <?php 
                foreach ($kayus as $j => $kayu_j): 
                    $value = $dominant['E'][$i][$j];
                ?>
                    <td><?php echo $value; ?></td>
                <?php endforeach; ?>
            </tr>
        <?php endforeach; ?>
    </tbody>
</table>

<!-- Hasil Akhir -->
<h3>Hasil Akhir (Kesimpulan)</h3>
<div class="conclusion">
    <p>Matrik E memberikan urutan pilihan dari setiap alternatif, yaitu bila ekl=1 maka alternatif Ak merupakan alternatif yang lebih baik dari Al, sehingga baris dalam matrik E yang memiliki ekl=1 paling sedikit dapat dieliminasi dan yang memiliki ekl terbanyak merupakan alternatif terbaik.</p>
    
    <p>Sehingga disimpulkan bahwa alternatif terbaik adalah:</p>
    
    <?php 
    $winner = array_keys($scores, max($scores))[0];
    ?>
    <p><strong>K00<?php echo $winner; ?>-<?php echo htmlspecialchars($kayus[$winner]['nama_kayu']); ?></strong> dengan total nilai e=<?php echo $scores[$winner]; ?></p>
</div>

<style>
.conclusion {
    background-color: #f8f9fa;
    padding: 20px;
    border-radius: 5px;
    margin: 20px 0;
    border-left: 4px solid #007bff;
}

.dashboard-table {
    width: 100%;
    border-collapse: collapse;
    margin: 20px 0;
    font-size: 14px;
}

.dashboard-table th, .dashboard-table td {
    border: 1px solid #ddd;
    padding: 8px;
    text-align: center;
}

.dashboard-table th {
    background-color: #f2f2f2;
    font-weight: bold;
}

.dashboard-table tr:nth-child(even) {
    background-color: #f9f9f9;
}

h2, h3 {
    color: #333;
    margin-top: 30px;
    margin-bottom: 15px;
}
</style>

<?php
$stmt->close();
include "../includes/footer.php";
?>
